/**
 * 
 */
package gov.va.med.mhv.usermgmt.util;

import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Properties;

import javax.management.ObjectName;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.config.ConfigurationManager;

import weblogic.management.runtime.ServerSecurityRuntimeMBean;
import weblogic.management.security.authentication.UserLockoutManagerMBean;

import gov.va.med.mhv.core.jmx.JMXServerConnection;
import gov.va.med.mhv.core.jmx.JMXServerConnectionFactory;
import gov.va.med.mhv.core.jmx.JMXServerType;
import gov.va.med.mhv.core.jmx.ObjectNameUtil;
import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.core.util.PropertiesUtil;

/**
 * Manages user lockout and password expiration.
 * Note that this class operates as a facade for WebLogic's 
 * UserLockoutManagerMBean.
 * @author Rob Proper
 */
public final class AccountSecurityManager {

	// NOTE: Could access UserLockoutManagerRuntimeMBean using:
	//   ServerRuntime.theOne().getServerSecurityRuntime().
	//		getDefaultRealmRuntime().getUserLockoutManagerRuntime().
	//      isLockedOut(userName);
	// However, this could not be used by unlockAccount, because not privileged 
	// to access other than read-only. To keep access consistent, using 
	// JMXConnection instead.
	

	private static final Log LOG = LogFactory.getLog(AccountSecurityManager.
		class);

	private static final String RESOURCE_NAME = "/account-security.properties";

	private static final String LOCKOUT_MANAGER_NAME = 
		"account.security.lockoutmanager.name";

	private static final String PASSWORD_EXPIRATION_PERIOD = 
		"account.security.password.expiration.period";

	private static final String SECURITY_REALM_NAME = 
		"account.security.realm.name";

	private static String DEFAULT_SECURITY_REALM_NAME = "myrealm";

	private static String DEFAULT_MBEAN_NAME = 
		"Security:Name=" + DEFAULT_SECURITY_REALM_NAME + "UserLockoutManager";

	private static int DEFAULT_LOCKOUT_PERIOD = 30;

	private static int DEFAULT_LOCKOUT_THRESHOLD = 5;

	private static int DEFAULT_PASSWORD_EXPIRATION_PERIOD = 24;

	private static final AccountSecurityManager instance = 
		new AccountSecurityManager();
	
	/**
	 * Returns a UserLockoutManager instance.
	 * @return a UserLockoutManager instance.
	 */
	public static final AccountSecurityManager getInstance() {
		return instance;
	}

	private final int passwordExpirationPeriod;
	private final int lockoutThreshold;
	private final int lockoutPeriod;
	private final String realmName; 
	private final String lockoutManagerName;
	private final ObjectName beanName;

	/**
	 * Hide constructor for singleton pattern
	 */
	private AccountSecurityManager() {

		Properties properties = null; 
		try {
			properties = ConfigurationManager.getConfiguration(RESOURCE_NAME);
		} catch (IllegalStateException e) {
			LOG.warn("Using defaults, because failed to load properties from '" 
				+ RESOURCE_NAME + "'.");
		}
		passwordExpirationPeriod = PropertiesUtil.getIntValue(RESOURCE_NAME, 
			properties, PASSWORD_EXPIRATION_PERIOD, 
			DEFAULT_PASSWORD_EXPIRATION_PERIOD);

		realmName = PropertiesUtil.getValue(RESOURCE_NAME, properties, 
			SECURITY_REALM_NAME, DEFAULT_SECURITY_REALM_NAME);
		lockoutManagerName = PropertiesUtil.getValue(RESOURCE_NAME, properties, 
			LOCKOUT_MANAGER_NAME, null);

		beanName = createManagerObjectName();
		
		int lockoutThreshold = DEFAULT_LOCKOUT_THRESHOLD;
		int lockoutPeriod = DEFAULT_LOCKOUT_PERIOD;
		UserLockoutManagerMBean lockoutManager = null;
		JMXServerConnection connection = connect(JMXServerType.DomainRuntime);
		if (connection != null) {
			try {
				lockoutManager = getManager(connection);
				if (lockoutManager != null) {
					lockoutThreshold = toInt(lockoutManager.
						getLockoutThreshold(), DEFAULT_LOCKOUT_THRESHOLD);
					lockoutPeriod = toInt(lockoutManager.
						getLockoutDuration(), DEFAULT_LOCKOUT_PERIOD);
				}
			} finally {
				close(connection);
			}
		}
		
		this.lockoutThreshold = lockoutThreshold;
		this.lockoutPeriod = lockoutPeriod;
		
		if (LOG.isInfoEnabled()) {
			LOG.info(((lockoutManager != null) ? "LockoutManager" : "default")
				+ "=["
				+ "Lockout Threshold:" + getLockoutThreshold()
				+ "; Lockout Period(min):" + getLockoutPeriod()
				+ "]");
			LOG.info(((properties  != null) ? "Configuration" : "default") 
				+ "=["
				+ "password Expiration Period(hours):" 
					+ getPasswordExpirationPeriod()
				+ "; realmName:" + getRealmName()
				+ "; lockoutManagerName:" + this.lockoutManagerName
				+ "]");
		}
	}

	/**
	 * Creates the ObjectName instance used to lookup the 
	 * LockoutUserManagerMBean.
	 * @return The ObjectName of LockoutUserManagerMBean. 
	 */
	private ObjectName createManagerObjectName() {
		String name = lockoutManagerName;
		if (StringUtils.isBlank(name)) {
			name = "Security:Name=" +  realmName + "UserLockoutManager";
			LOG.info("'lockoutManagerName' is not configured. Using '"
				+ name); 
		}
		return ObjectNameUtil.createBeanName(name, DEFAULT_MBEAN_NAME);
	}
	
	/**
	 * Unlocks a give user account. 
	 * @param userName The account to unlock.
	 */
	public void unlockAccount(String userName) {
		Precondition.assertNotBlank("userName", userName);
		if (LOG.isDebugEnabled()) {
			LOG.debug("Unlocking account '" + userName+ "'");
		}
		JMXServerConnection connection = connect(JMXServerType.Runtime);
		if (connection != null) {
			try {
                ServerSecurityRuntimeMBean lockoutManager = 
					getRuntimeManager(connection);
				if (lockoutManager != null) {
					lockoutManager.clearLockout(userName);
					if (LOG.isDebugEnabled()) {
						LOG.debug("Unlocked account '" + userName+ "'");
					}
				}
			} finally {
				close(connection);
			}
		} else {
			LOG.error("Unable to unlock user account '" + userName + "'");
		}
	}

	public boolean isAccountLocked(String userName) {
		Precondition.assertNotBlank("userName", userName);
		boolean isLocked = false;
		JMXServerConnection connection = connect(JMXServerType.Runtime);
		if (connection != null) {
			try {
                ServerSecurityRuntimeMBean lockoutManager = 
					getRuntimeManager(connection);
				if (lockoutManager != null) {
					isLocked = lockoutManager.isLockedOut(userName);
                    if (isLocked) {
                        // This is a workaround, for WebLogic not release
                        // the log upon expiration, so here we check if
                        // the lock has expired
                        final long MILLISECONDS_PER_MINUTE = 60 * 1000; 
                        long lastFailure = lockoutManager.getLastLoginFailure(
                            userName); 
                        long expires = lastFailure + getLockoutPeriod() 
                            * MILLISECONDS_PER_MINUTE;
                        long now = System.currentTimeMillis();
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("last    = " + lastFailure 
                                  + "\nexpires = " + expires 
                                  + "\nnow     = " + now
                                  + "\nexpired = " + (expires <= now));
                        }
                        isLocked = (now < expires);
                    }
				}
			} finally {
				close(connection);
			}
		} else {
			LOG.error("Unable to check if user account '" + userName 
				+ "' is locked. Asumming it is not locked.");
		}
		if (LOG.isDebugEnabled()) {
			LOG.debug("Account '" + userName + "' is "
				+ ((isLocked) ? "" : "NOT ") + "locked");
		}
		return isLocked;
	}
	
	/**
	 * Determines if a user account password has expired on a given date.
	 * @param lastUpdatedTime The date to check account expiration for.
	 * @return True if the account has expired; False otherwise.
	 */
	public boolean hasAccountExpired(Timestamp lastUpdatedTime) {
		Precondition.assertNotNull("time", lastUpdatedTime);
		Calendar calendar = Calendar.getInstance();
		calendar.add(Calendar.HOUR, - getPasswordExpirationPeriod());
		Timestamp loginWindowStart = new Timestamp(calendar.getTimeInMillis());
		// last updated last before start of period
		return lastUpdatedTime.before(loginWindowStart);
	}

	/** 
	 * Denotes the time (in minutes) that a user account is locked down for.
	 * An account is locked down after the maximum number of login failures has 
	 * been reached.   
	 * @return The numer of minutes that account is locked down.
	 * @see #getLockoutThreshold()
	 */
	public int getLockoutPeriod() {
		
		return lockoutPeriod;
	}

	/** 
	 * Denotes the number of allowed login failures before a user account is 
	 * locked.   
	 * @return The number of allowed login failures.
	 */
	public int getLockoutThreshold() {
		return lockoutThreshold;
	}
	
	/**
	 * Make a connection to the JMX domain server.
	 * @return The connection.
	 */
	private JMXServerConnection connect(JMXServerType serverType) {
		JMXServerConnection connection = null;
		try {
			connection = JMXServerConnectionFactory.getInstance().
				getConnection(serverType);
			if (connection != null) {
				connection.connect();
			}
		} catch (Exception e) {
			LOG.error("Unable to get JMX server connection", e);
		}
		return connection;
	}

	/**
	 * Close the given JMX server connection (if any). 
	 * @param connection The connection to close.
	 */
	private void close(JMXServerConnection connection) {
		if (connection != null) {
			connection.close();
		}
	}

	/**
	 * Gets the UserLockoutManagerMBean through a given JMX connection.
	 * @param connection The connection
	 * @return The UserLockoutManagerMBean, if found; null otherwise.
	 */
	private UserLockoutManagerMBean getManager(JMXServerConnection connection) {
		UserLockoutManagerMBean lockoutManager = null;
		try {
			lockoutManager = (UserLockoutManagerMBean) connection.getMBean(
				beanName, UserLockoutManagerMBean.class);
			if (lockoutManager == null) {
				LOG.warn("Unable to find LockoutManagerMBean using '"
					+ beanName + "'");
			}
		} catch(Exception e) {
			LOG.error("Unable to access LockoutManagerMBean using '"
				+ beanName + "'", e);
		}
		return lockoutManager;
	}

	/**
	 * Gets the UserLockoutManagerMBean through a given JMX connection.
	 * @param connection The connection
	 * @return The UserLockoutManagerMBean, if found; null otherwise.
	 */
	private ServerSecurityRuntimeMBean getRuntimeManager(
		JMXServerConnection connection) 
	{
        ServerSecurityRuntimeMBean lockoutManager = null;
		try {
			lockoutManager = (ServerSecurityRuntimeMBean) connection.
				getMBean(beanName, ServerSecurityRuntimeMBean.class);
			if (lockoutManager == null) {
				LOG.warn("Unable to find ServerSecurityRuntimeMBean using '"
					+ beanName + "'");
			}
		} catch(Exception e) {
			LOG.error("Unable to access ServerSecurityRuntimeMBean using '"
				+ beanName + "'", e);
		}
		return lockoutManager;
	}

	/**
	 * Denotes the time after which a temporary password expires.
	 * @return The number of hours after which a temporary password expires.
	 */
	public int getPasswordExpirationPeriod() {
		return passwordExpirationPeriod;
	}

	/**
	 * The number of the security realm used for account authentication.
	 * @return Te security realm name.
	 */
	public String getRealmName() {
		return realmName;
	}
	
	private int toInt(Long value, int defaultValue) {
		return (value != null) ? value.intValue() : defaultValue;
	}

}
